// Copyright 2023 Tencent Inc.  All rights reserved.
/**
 * @file split_string_by_absl.cc
 * @brief 使用 google absl 库做字符串分割
 * 安装方法参考 https://abseil.io/docs/cpp/quickstart-cmake：
 * 1、git clone https://git.woa.com/github-mirrors/abseil/abseil-cpp.git
 * 2、cd absei-cpp && makdie build_dir && cd build_dir
 * 3、cmake .. -DCMAKE_CXX_STANDARD=20 -DCMAKE_INSTALL_PREFIX=/usr/local -DCMAKE_CXX_FLAGS=-fPIC
 * 4、make -j8
 * 5、sudo make install
 *
 * 总结：
 * 1、absl strings 库支持单字符和多字符分割
 * 2、在长字符串分割的case下，性能不如自己实现的
 *
 * 对长文本的评测结果：
./split_string_by_char 结果：
simd cost_ms:413
simd V2 cost_ms:501
no simd V1 cost_ms:552
no simd V2 cost_ms:424
no simd V3 cost_ms:508

./split_string_by_string 结果：
not simd cost_ms:752
not simd V2 cost_ms:673
simd cost_ms:484

 ./split_string_by_absl 结果：
absl by one char cost_ms:709
absl by any char cost_ms:1054

上述评测显示，
（1）单字符分割，性能比前述实现的最差的版本还差；
（2）多字符分割，性能比前述实现的最差的版本还差；
 怀疑和absl的字符串分割会产出空子串有关系，这会浪费一次std::string_view的构造。
 */

#include <bitset>
#include <chrono>
#include <iostream>
#include <string>
#include <vector>

#include "absl/strings/str_split.h"

int main() {
  std::vector<std::string_view> sentences;
  //  sentences.emplace_back(",  ,,,,,,,");
  // sentences.emplace_back(",, ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,");
  // sentences.emplace_back(",,, ,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,,");
  // sentences.emplace_back("1111111111");
  // sentences.emplace_back("1111111111111111111111111111111111111");
  // sentences.emplace_back("11 1111111111111   11111111111111111111111 ,11");
  //  sentences.emplace_back("11 11 11,111,111,111,111,1,1,1,,1,11  11111111111111,1111111111");
  sentences.emplace_back(
      "11 11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11 11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,1111111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1111111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "11,11,11,111,111,111,111,1,1,1,,1,11,1,1111111111111,1,1,11111111,"
      "111111111111111111111111111111111111111111,111111111");

  // split string by absl one char
  {
    auto start_commit = std::chrono::steady_clock::now().time_since_epoch();
    for (int i = 0; i != 10000; ++i) {
      for (const auto& item : sentences) {
        std::vector<std::string_view> ret = absl::StrSplit(item, ',');
      }
    }
    auto end_commit = std::chrono::steady_clock::now().time_since_epoch();
    std::cout << "absl by one char cost_ms:"
              << std::chrono::duration_cast<std::chrono::milliseconds>(end_commit - start_commit).count() << std::endl;
  }

  // split string by absl any char
  {
    auto start_commit = std::chrono::steady_clock::now().time_since_epoch();
    for (int i = 0; i != 10000; ++i) {
      for (const auto& item : sentences) {
        std::vector<std::string_view> ret = absl::StrSplit(item, absl::ByAnyChar("\t ,"));
      }
    }
    auto end_commit = std::chrono::steady_clock::now().time_since_epoch();
    std::cout << "absl by any char cost_ms:"
              << std::chrono::duration_cast<std::chrono::milliseconds>(end_commit - start_commit).count() << std::endl;
  }

  return 0;
}